const Pusher = require("pusher-js/node");
const axios = require("axios");

const appKey = "leaderos-connect";
const host = "connect-socket.leaderos.net";
const authEndpoint = "https://connect-api.leaderos.net/broadcasting/auth";
const queueFile = "queue.json";
const resourceName = GetCurrentResourceName();

// 🌐 Localized messages
const lang = config.lang;
const messages = config.messages;

// Website URL validation
let websiteUrl = config.websiteUrl;
if (websiteUrl === "") {
    console.error(messages[lang].error("Website URL is not set in config.js"));
}
if (websiteUrl.startsWith("http://")) {
    console.error(messages[lang].error("Website URL must use HTTPS protocol"));
}
if (websiteUrl.endsWith("/")) {
  websiteUrl = websiteUrl.slice(0, -1);
}

// Validate API key
if (!config.apiKey || config.apiKey === "") {
    console.error(messages[lang].error("API key is not set in config.js"));
}

// Validate server token
if (!config.serverToken || config.serverToken === "") {
    console.error(messages[lang].error("Server token is not set in config.js"));
}

// 🗂️ Read queue JSON
function readQueue() {
  try {
    const content = LoadResourceFile(resourceName, queueFile);
    return content ? JSON.parse(content) : {};
  } catch (err) {
    console.error(messages[lang].error("Queue read failed"));
    return {};
  }
}

// 💾 Write queue JSON
function writeQueue(data) {
  try {
    SaveResourceFile(resourceName, queueFile, JSON.stringify(data, null, 2));
  } catch (err) {
    console.error(messages[lang].error("Queue write failed"));
  }
}

// ▶️ Execute commands
function executeCommands(commands, uniqID, sid = null) {
  commands.forEach((cmd) => {
    // 🧩 Replace %sid% with actual player ID
    if (sid !== null) {
      cmd = cmd.replace(/%sid%/g, sid.toString());
    }
    // 🏃‍♂️ Log and execute command
    console.log(messages[lang].running(uniqID, cmd));

    // Execute the command
    ExecuteCommand(cmd);
  });
}

// 🔍 Find player by identifier
function getPlayerByIdentifier(uniqID) {
  for (let i = 0; i < GetNumPlayerIndices(); i++) {
    const player = GetPlayerFromIndex(i);
    const identifiers = getPlayerIdentifiers(player);
    for (const id of identifiers) {
      if (id === uniqID) return player;
    }
  }
  return null;
}

// 🔑 Get player identifiers
function getPlayerIdentifiers(src) {
  const ids = [];
  for (let i = 0; i < GetNumPlayerIdentifiers(src); i++) {
    ids.push(GetPlayerIdentifier(src, i));
  }
  return ids;
}

// 🚀 Initialize on resource start
on("onServerResourceStart", (resourceName) => {
  if (resourceName !== GetCurrentResourceName()) return;

  const pusher = new Pusher(appKey, {
    cluster: "eu",
    wsHost: host,
    wsPort: 6001,
    wssPort: 6002,
    forceTLS: true,
    disableStats: true,
    enabledTransports: ["ws", "wss"],
    authEndpoint: authEndpoint,
    auth: {
      headers: {
        'X-Api-Key': config.apiKey,
        'Accept': 'application/json'
      },
    },
  });

  const channelName = `private-servers.${config.serverToken}`;
  const channel = pusher.subscribe(channelName);

  channel.bind("pusher:subscription_succeeded", () => {
    console.log(messages[lang].connected);
  });

  channel.bind("send-commands", async (data) => {
    try {
      const payload = data;
      const commandIDs = payload.commands;

      const formData = new URLSearchParams();
      formData.append("token", config.serverToken);
      commandIDs.forEach((id, i) => {
        formData.append(`commands[${i}]`, id);
      });

      const res = await axios.post(`${config.websiteUrl}/api/command-logs/validate`, formData, {
        headers: {
          'X-Api-Key': config.apiKey,
          'Accept': 'application/json',
          'Content-Type': 'application/x-www-form-urlencoded',
          'User-Agent': 'LeaderOS-FiveM/1.0'
        }
      });

      const validatedCommands = res.data.commands;
      let uniqID = "";
      let commands = [];

      for (const item of validatedCommands) {
        if (item.command) {
          commands.push(item.command);
          if (!uniqID) {
            if (item.authProvider === "steam") {
              uniqID = `steam:${item.username}`;
            } else if (item.authProvider === "discord") {
              uniqID = `discord:${item.username}`;
            } else if (item.authProvider === "cfx") {
              uniqID = `fivem:${item.username}`;
            } else {
              uniqID = item.username;
            }
          }
        }
      }

      if (config.checkPlayerOnline) {
        const player = getPlayerByIdentifier(uniqID);
        if (player !== null) {
          executeCommands(commands, uniqID, player);
        } else {
          const queue = readQueue();
          queue[uniqID] = queue[uniqID] || [];
          queue[uniqID].push(...commands);
          writeQueue(queue);
          commands.forEach(command => {
            console.log(messages[lang].queued(uniqID, command));
          });
        }
      } else {
        executeCommands(commands, uniqID);
      }

    } catch (error) {
      console.error(messages[lang].error(error.message));
    }
  });

  channel.bind("ping", () => {
    console.log(messages[lang].ping);
  });
});

// Only check player join event if checkPlayerOnline is true
if (config.checkPlayerOnline) {
  // 👤 On player join, run queued commands
  on("playerJoining", () => {
    const playerID = source; // 🎯 Player's server-side session ID (source)
    const identifiers = getPlayerIdentifiers(playerID);
    const queue = readQueue();

    for (const id of identifiers) {
      if (queue[id]) {
        const commands = queue[id];
        executeCommands(commands, id, playerID);
        delete queue[id];
        writeQueue(queue);
        console.log(messages[lang].joined(id));
      }
    }
  });
}
